<?php

namespace App\Http\Controllers\API\Auth;

use App\Http\Resources\Customer\CustomerResource;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class CustomerAuthController extends Controller
{
    public function login(Request $request): JsonResponse
    {
        $data = $request->all();

        $validator = Validator::make($data, [
            'phone' => ['required', 'digits:10'],
            'password' => ['required', 'string'],
        ]);

        if ($validator->fails()) {
            return response()
                ->json(['error' => $validator->errors()->first()], 422);
        }

        $customer = User::query()->role('customer')->wherePhone($data['phone'])->first();

        if (!$customer || !Hash::check($data['password'], $customer->password)) {
            return response()->json([
                'status' => false,
                'message' => 'The provided credentials are incorrect.'
            ], 401);
        }

        return response()->json([
            'token' => $customer->createToken($data['phone'], ['role:customer'])->plainTextToken,
            'user' => CustomerResource::make($customer),
        ]);
    }

    public function register(Request $request): JsonResponse
    {
        $data = $request->all();

        $validator = Validator::make($data, [
            'name' => ['required', 'string'],
            'phone' => ['required', 'digits:10', 'unique:users'],
            'email' => ['email', 'unique:users'],
            'password' => ['required', 'string', 'min:6'],
        ]);

        if ($validator->fails()) {
            return response()
                ->json(['error' => $validator->errors()->first()], 422);
        }

        $data['password'] = Hash::make($data['password']);

        /** @var User $customer */
        $customer = User::query()->create($data);

        $customer->assignRole('customer');
        $customer->wallet()->create();

        return response()->json([
            'token' => $customer->createToken($data['phone'], ['role:customer'])->plainTextToken,
            'user' => CustomerResource::make($customer),
        ]);
    }

    public function logout()
    {
        auth()->user()->currentAccessToken()->delete();
    }
}
