<?php

namespace App\Http\Controllers\API\Customer\Checkout;

use App\Http\Controllers\Controller;
use App\Mail\OrderShipped;
use App\Models\Carts\Cart;
use App\Models\Orders\Order;
use App\Models\Packages\Basket;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class WebhookController extends Controller
{
    public function handle(Request $request): JsonResponse
    {
        $orderId = $request->orderId;
        $order = Order::query()
            ->where('order_id', $orderId)
            ->get()
            ->first();

        if ($order == null) return response()->json(['status' => 'error']);

        if ($order->payment_method != Order::PM_BANK) {
            $this->updateOrder($order);
            $this->creditPackage($order);

            return response()->json(['status' => 'success']);
        }

        if ($order->verified) return response()->json(['status' => 'Already Verified!']);

        if (!$this->verifySignature($request, $order)) {
            return response()->json(['error' => 'Unable to verify response!'], 400);
        }

        $referenceId = $request->referenceId;

        if ($request->txStatus == 'FAILED') {
            $order->update([
                'verified' => true,
                'payment_id' => $referenceId,
                'payment_status' => Order::PAYMENT_FAILED,
                'status' => Order::ORDER_FAILED,
            ]);

            return response()->json(['error' => 'Unable to pay!']);
        }

        $order->update([
            'verified' => true,
            'payment_id' => $referenceId,
            'payment_status' => Order::PAYMENT_SUCCESS,
            'status' => Order::ORDER_CONFIRMED,
        ]);

        try {
            Mail::to($order->customer)->send(new OrderShipped($order));
        }catch (\Exception $e) {

        }

        $this->creditPackage($order);

        if ($order->type == Order::TYPE_DEBT) {
            $this->transaction->undebt($order->customer_id, $order->total['amount']);
        } elseif ($order->type == Order::TYPE_WALLET) {
            $this->transaction->credit($order->customer_id, $order->total['amount']);
        } else {
            Cart::query()
                ->where('customer_id', $order->customer_id)
                ->delete();
        }

        return response()->json(['status' => 'success']);
    }

    protected function updateOrder(Order $order)
    {
        $order->update([
            'verified' => true,
            'payment_id' => '',
            'payment_status' => Order::PAYMENT_SUCCESS,
            'status' => Order::ORDER_CONFIRMED,
        ]);

        try {
            Mail::to($order->customer)->send(new OrderShipped($order));
        }catch (\Exception $e) {

        }

        Cart::query()
            ->where('customer_id', $order->customer_id)
            ->delete();
    }

    protected function creditPackage(Order $order)
    {
        if ($order->type != Order::TYPE_BASKET) return;

        $basket = Basket::query()->find($order->basket_id);
        $amount = $order->total['amount'] + ($order->total['amount'] * $basket->discount_percentage) / 100;

        $this->transaction->creditPackage($order->customer_id, $amount);
    }

    protected function verifySignature(Request $request, Order $order): bool
    {
        $referenceId = $request->referenceId;
        $txStatus = $request->txStatus;

        $orderId = $request->orderId;

        $orderAmount = $request->orderAmount;
        $paymentMode = $request->paymentMode;
        $txMsg = $request->txMsg;
        $txTime = $request->txTime;
        $signature = $request->signature;

        $data = $orderId . $orderAmount . $referenceId . $txStatus . $paymentMode . $txMsg . $txTime;
        $hash_hmac = hash_hmac('sha256', $data, config('cashfree.secret'), true);
        $computedSignature = base64_encode($hash_hmac);

        if ($signature != $computedSignature) {
            $order->update([
                'payment_id' => $referenceId,
                'payment_status' => Order::PAYMENT_FAILED,
                'status' => Order::ORDER_FAILED,
            ]);

            return false;
        }

        return true;
    }
}
