<?php

namespace App\Http\Livewire;

use App\Models\Brands\Brand;
use App\Models\Categories\Category;
use App\Models\Orders\Order;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;
use Rappasoft\LaravelLivewireTables\DataTableComponent;
use Rappasoft\LaravelLivewireTables\Views\Column;
use Rappasoft\LaravelLivewireTables\Views\Filter;

class OrdersTable extends DataTableComponent
{

    public function columns(): array
    {
        return [
            Column::make('ORDER ID', 'order_id')->searchable(),

            Column::make('CUSTOMER', 'customer.name')->format(function ($value) {
                return view('admin.shared.avatar')->with(['name' => $value]);
            })->searchable(),

            Column::make('TOTAL', 'total')->format(function ($value) {
                return '<span class="text-green">' . $value['currency'] . '</span>';
            })->asHtml(),

            Column::make('TIME', 'created_at')->format(function ($value) {
                return $value->diffForHumans();
            }),

            Column::make('CHOICE', 'women_delivery')->format(function ($value) {
                if ($value) return '<i class="fas fa-female" style="color: #601FCC"></i>';

                return '<i class="fas fa-male mr-1"></i><i class="fas fa-female"></i>';
            })->asHtml(),

            Column::make('DELIVERIES')->format(function ($value, $column, $row) {
                return view('admin.order.delivery', compact('row'));
            }),

            Column::make('TYPE', 'type')->format(function ($value, $column, $row) {
                return view('admin.order.type', compact('row'));
            }),

            Column::make('PAYMENT METHOD', 'payment_method')->format(function ($value) {
                return Str::upper($value);
            }),

            Column::make('PAYMENT', 'payment_status')->format(function ($value, $column, $row) {
                return view('admin.order.payment', compact('row'));
            }),

            Column::make('STATUS', 'status')->format(function ($value, $column, $row) {
                return view('admin.order.status', compact('row'));
            }),

            Column::make('CONFIRMED', 'confirmed')->format(fn($value) => $value == 1 ? 'TRUE' : 'FALSE'),

            Column::make('ACTIONS')->format(function ($value, $column, $row) {
                return view('admin.order.action', compact('row'));
            }),
        ];
    }

    public function filters(): array
    {
        $all = ['' => 'All'];

        $status = [
            Order::ORDER_PENDING => 'PENDING',
            Order::ORDER_CONFIRMED => 'CONFIRMED',
            Order::ORDER_FAILED => 'FAILED',
            Order::ORDER_ACTIVE => 'ACTIVE',
            Order::ORDER_COMPLETED => 'COMPLETED',
        ];

        $types = [
            Order::TYPE_CART => 'CART',
            Order::TYPE_PACKAGE => 'PACKAGE',
            Order::TYPE_BASKET => 'BASKET',
            Order::TYPE_WALLET => 'WALLET',
            Order::TYPE_DONATION => 'DONATION',
        ];

        $payments = [
            Order::PAYMENT_FAILED => 'FAILED',
            Order::PAYMENT_SUCCESS => 'SUCCESS',
            Order::PAYMENT_DUE => 'DUE',
        ];

        return [
            'status' => Filter::make('Status')->select(array_replace($all, $status)),
            'type' => Filter::make('Type')->select(array_replace($all, $types)),
            'payment' => Filter::make('Payment')->select(array_replace($all, $payments)),
            'from' => Filter::make('From')->date(),
            'to' => Filter::make('To')->date(),
        ];
    }

    public function query(): Builder
    {
        return Order::query()
            ->latest()
            ->has('customer')
            ->with('customer')
            ->withCount('deliveries')
            ->when($this->getFilter('status'), function ($query, $status) {
                return $query->where('status', $status);
            })
            ->when($this->getFilter('type'), function ($query, $type) {
                return $query->where('type', $type);
            })
            ->when($this->getFilter('payment'), function ($query, $payment) {
                return $query->where('payment_status', $payment);
            })
            ->when($this->getFilter('from'), function ($query, $date) {
                return $query->where('created_at', '>=', $date);
            })
            ->when($this->getFilter('to'), function ($query, $date) {
                return $query->where('created_at', '<=', $date);
            })
            ->latest();
    }
}
