<?php

namespace App\Services;

use App\Exceptions\WalletException;
use App\Models\User;
use App\Models\Wallets\Wallet;


class TransactionService
{
    protected WalletService $wallet;

    public function __construct(WalletService $wallet)
    {
        $this->wallet = $wallet;
    }

    public function credit(int $id, $amount)
    {
        $customer = $this->user($id);
        $wallet = $customer->wallet;

        $customer->transactions()->create([
            'amount' => $amount,
            'type' => 'credit',
            'wallet_id' => $wallet->id,
        ]);

        $this->wallet->credit($wallet, $amount);
    }

    /**
     * @throws WalletException
     */
    public function debit(int $id, $amount): int
    {
        $customer = $this->user($id);
        $wallet = $customer->wallet;

        $balance = $wallet->balance['amount'];

        if ($amount > $balance) {
            $debt = $wallet->debt['amount'];
            $maxDebt = $wallet->max_debt['amount'];

            $extraAmount = $amount - $balance;

            if (($debt + $extraAmount) >= $maxDebt) throw  new WalletException();

            $this->wallet->debt($wallet, $extraAmount);
            $wallet->update(['balance' => 0]);
        } else {
            $this->wallet->debit($wallet, $amount);
        }

        return $customer->transactions()->create([
            'amount' => $amount,
            'type' => 'debit',
            'wallet_id' => $wallet->id,
        ])->id;
    }

    public function creditPackage(int $id, $amount)
    {
        $customer = $this->user($id);
        $wallet = $customer->wallet;

        $customer->transactions()->create([
            'amount' => $amount,
            'type' => 'credit',
            'wallet_id' => $wallet->id,
        ]);

        $this->wallet->creditPackage($wallet, $amount);
    }

    /**
     * @throws WalletException
     */
    public function debitPackage(int $id, $amount): int
    {
        $customer = $this->user($id);
        $wallet = $customer->wallet;

        $balance = $wallet->package['amount'];

        if ($amount > $balance) {
            $debt = $wallet->debt['amount'];
            $maxDebt = $wallet->max_debt['amount'];

            $extraAmount = $amount - $balance;

            if (($debt + $extraAmount) >= $maxDebt) throw  new WalletException();

            $this->wallet->debt($wallet, $extraAmount);
            $wallet->update(['package' => 0]);
        } else {
            $this->wallet->debitPackage($wallet, $amount);
        }

        return $customer->transactions()->create([
            'amount' => $amount,
            'type' => 'debit',
            'wallet_id' => $wallet->id,
        ])->id;
    }

    /**
     * @throws WalletException
     */
    public function debitCoupon(int $id, $amount): int
    {
        $customer = $this->user($id);
        $wallet = $customer->wallet;

        $balance = $wallet->coupon['amount'];

        if ($balance < $amount) throw  new WalletException();

        $this->wallet->debitCoupon($wallet, $amount);

        return $customer->transactions()->create([
            'amount' => $amount,
            'type' => 'debit',
            'wallet_id' => $wallet->id,
        ])->id;
    }

    public function reserve(int $id, $amount)
    {
        $customer = $this->user($id);
        $wallet = $customer->wallet;

        $customer->transactions()->create([
            'amount' => $amount,
            'type' => 'debit',
            'wallet_id' => $wallet->id,
        ]);

        $this->wallet->reserve($wallet, $amount);
    }

    public function unreserve(int $id, $amount)
    {
        $customer = $this->user($id);
        $wallet = $customer->wallet;

        $customer->transactions()->create([
            'amount' => $amount,
            'type' => 'credit',
            'wallet_id' => $wallet->id,
        ]);

        $this->wallet->unreserve($wallet, $amount);
    }

    public function transfer(int $senderId, int $receiverId, $amount)
    {
        $sender = $this->user($senderId);
        $senderWallet = $sender->wallet;

        $receiver = $this->user($receiverId);
        $receiverWallet = $receiver->wallet;

        $this->wallet->debit($senderWallet, $amount);
        $this->wallet->credit($receiverWallet, $amount);

        $sender->transactions()->create([
            'amount' => $amount,
            'type' => 'debit',
            'wallet_id' => $senderWallet->id,
        ]);

        $sender->transactions()->create([
            'amount' => $amount,
            'type' => 'credit',
            'wallet_id' => $receiverWallet->id,
        ]);
    }

    public function coupon(int $id)
    {
        $customer = $this->user($id);
        $wallet = $customer->wallet;

        $customer->transactions()->create([
            'amount' => $wallet->coupon['amount'],
            'type' => 'credit',
            'wallet_id' => $wallet->id,
        ]);

        $this->wallet->coupon($wallet);
    }

    protected function user(int $id): User
    {
        return User::query()->where('id', $id)->with('wallet')->first();
    }

    public function recharge($amount, Wallet $wallet)
    {
        $customer = $this->user(auth()->id());

        $customer->transactions()->create([
            'amount' => $amount,
            'type' => 'credit',
            'wallet_id' => $wallet->id,
        ]);

        $this->wallet->credit($wallet, $amount);
    }

    public function undebt(int $id, $amount): int
    {
        $customer = $this->user($id);

        $this->wallet->undebt($customer->wallet, $amount);

        return 1;
    }
}
